
"use client";

import * as React from "react";
import Link from "next/link";
import { usePathname } from 'next/navigation';
import {
  Sidebar,
  SidebarHeader,
  SidebarContent,
  SidebarMenu,
  SidebarMenuItem,
  SidebarMenuButton,
  SidebarFooter,
  useSidebar,
  SidebarProvider,
} from "@/components/ui/sidebar";
import {
  Home,
  Bot,
  DollarSign,
  ShoppingCart,
  Zap,
  LifeBuoy,
  LogOut,
  Settings,
  Menu,
  PlusCircle,
  ShoppingBag,
  User,
  UserCheck,
} from "lucide-react";
import { Logo } from "./Logo";
import { Avatar, AvatarFallback, AvatarImage } from "../ui/avatar";
import { Button } from "../ui/button";
import { ThemeSwitcher } from "./ThemeSwitcher";
import { LanguageSwitcher } from "./LanguageSwitcher";
import { AgentTypeSelectionDialog } from "@/components/agent/AgentTypeSelectionDialog";

interface AppLayoutProps {
    children: React.ReactNode;
    translations: {
        dashboard: string;
        myAgents: string;
        sales: string;
        purchases: string;
        myAccount: string;
        support: string;
        marketplace: string;
        createAgent: string;
        settings: string;
        logout: string;
        affiliates: string;
    }
}

function AppSidebar({ translations }: { translations: AppLayoutProps['translations'] }) {
  const pathname = usePathname();
  const locale = pathname.split('/')[1] || 'pt';

  const menuItems = [
    { href: `/${locale}/dashboard`, label: translations.dashboard, icon: Home },
    { href: `/${locale}/my-agents`, label: translations.myAgents, icon: Bot },
    { href: `/${locale}/sales`, label: translations.sales, icon: DollarSign },
    { href: `/${locale}/purchases`, label: translations.purchases, icon: ShoppingCart },
    { href: `/${locale}/affiliates`, label: translations.affiliates, icon: UserCheck },
  ];

  return (
    <Sidebar>
      <SidebarHeader>
        <div className="flex items-center gap-2">
            <Logo />
            <span className="font-bold text-lg font-headline">Agenmax</span>
        </div>
      </SidebarHeader>
      <SidebarContent>
         <div className="p-2">
            <AgentTypeSelectionDialog
                trigger={
                    <Button className="w-full">
                        <PlusCircle className="mr-2 h-4 w-4"/>
                        {translations.createAgent}
                    </Button>
                }
                locale={locale}
             />
        </div>
        <SidebarMenu>
          {menuItems.map((item) => (
            <SidebarMenuItem key={item.href}>
              <SidebarMenuButton
                asChild
                isActive={pathname === item.href}
                tooltip={{
                  children: item.label,
                }}
              >
                <Link href={item.href}>
                  <item.icon />
                  <span>{item.label}</span>
                </Link>
              </SidebarMenuButton>
            </SidebarMenuItem>
          ))}
           <SidebarMenuItem key="marketplace">
              <SidebarMenuButton asChild isActive={pathname.includes('/marketplace')} tooltip={{children: translations.marketplace}}>
                    <Link href={`/${locale}/marketplace`}>
                        <ShoppingBag />
                        <span>{translations.marketplace}</span>
                    </Link>
                </SidebarMenuButton>
           </SidebarMenuItem>
        </SidebarMenu>
      </SidebarContent>
      <SidebarFooter>
         <SidebarMenu>
            <SidebarMenuItem key="my-account">
                 <SidebarMenuButton asChild isActive={pathname.includes('/settings')} tooltip={{children: translations.myAccount}}>
                    <Link href={`/${locale}/settings`}>
                        <User />
                        <span>{translations.myAccount}</span>
                    </Link>
                </SidebarMenuButton>
            </SidebarMenuItem>
            <SidebarMenuItem key="logout">
                 <SidebarMenuButton asChild tooltip={{children: translations.logout}}>
                    <Link href={`/${locale}`}>
                        <LogOut />
                        <span>{translations.logout}</span>
                    </Link>
                </SidebarMenuButton>
            </SidebarMenuItem>
        </SidebarMenu>
      </SidebarFooter>
    </Sidebar>
  );
}

function MobileHeader() {
    const { setOpenMobile } = useSidebar();
    const pathname = usePathname();
    const locale = pathname.split('/')[1] || 'pt';

    return (
        <header className="md:hidden flex items-center justify-between p-4 border-b">
             <Button variant="ghost" size="icon" onClick={() => setOpenMobile(true)}>
                <Menu />
            </Button>
            <Link href={`/${locale}/dashboard`} className="flex items-center gap-2">
              <Logo />
              <span className="font-bold font-headline">Agenmax</span>
            </Link>
             <div className="flex items-center gap-2">
              <LanguageSwitcher />
              <ThemeSwitcher />
              <Avatar className="h-9 w-9">
                  <AvatarImage src="https://placehold.co/40x40.png" alt="User" />
                  <AvatarFallback>U</AvatarFallback>
              </Avatar>
            </div>
        </header>
    )
}

function DesktopHeader() {
  return (
    <header className="hidden md:flex items-center justify-end p-4">
      <div className="flex items-center gap-4">
        <LanguageSwitcher />
        <ThemeSwitcher />
        <Avatar className="h-9 w-9">
          <AvatarImage src="https://placehold.co/40x40.png" alt="User" />
          <AvatarFallback>U</AvatarFallback>
        </Avatar>
      </div>
    </header>
  )
}

export function AppLayout({ children, translations }: AppLayoutProps) {
  return (
    <SidebarProvider>
        <div className="md:flex">
            <AppSidebar translations={translations} />
             <div className="flex-1 flex flex-col">
                <MobileHeader />
                <DesktopHeader />
                <main className="flex-1 p-4 md:p-8 pt-0 md:pt-0">
                    {children}
                </main>
            </div>
        </div>
    </SidebarProvider>
  );
}
