
"use client";

import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { CheckCircle, Copy, ExternalLink } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import Link from "next/link";
import { useState } from "react";
import { cn } from "@/lib/utils";

interface AgentCreatedDialogProps {
    open: boolean;
    onOpenChange: (open: boolean) => void;
    agentId: string;
    locale: string;
}

const translations = {
    pt: {
        title: "Agente Criado com Sucesso!",
        description: "Seu agente está pronto para ser compartilhado e vendido. Use os links abaixo.",
        checkoutLink: "Link do Checkout",
        checkoutDescription: "Compartilhe este link para que os clientes comprem seu agente.",
        embedCode: "Código de Incorporação",
        embedDescription: "Adicione o checkout diretamente ao seu site.",
        marketplaceLink: "Página no Marketplace",
        marketplaceDescription: "Veja como seu agente aparece para outros usuários.",
        copied: "Copiado!",
        copy: "Copiar",
        viewOnMarketplace: "Ver no Marketplace",
        close: "Fechar",
    },
    en: {
        title: "Agent Created Successfully!",
        description: "Your agent is ready to be shared and sold. Use the links below.",
        checkoutLink: "Checkout Link",
        checkoutDescription: "Share this link for customers to purchase your agent.",
        embedCode: "Embed Code",
        embedDescription: "Add the checkout directly to your website.",
        marketplaceLink: "Marketplace Page",
        marketplaceDescription: "See how your agent appears to other users.",
        copied: "Copied!",
        copy: "Copy",
        viewOnMarketplace: "View on Marketplace",
        close: "Close",
    },
    es: {
        title: "¡Agente Creado con Éxito!",
        description: "Tu agente está listo para ser compartido y vendido. Usa los enlaces a continuación.",
        checkoutLink: "Enlace de Checkout",
        checkoutDescription: "Comparte este enlace para que los clientes compren tu agente.",
        embedCode: "Código de Inserción",
        embedDescription: "Añade el checkout directamente a tu sitio web.",
        marketplaceLink: "Página del Marketplace",
        marketplaceDescription: "Mira cómo aparece tu agente para otros usuarios.",
        copied: "¡Copiado!",
        copy: "Copiar",
        viewOnMarketplace: "Ver en Marketplace",
        close: "Cerrar",
    }
};

const LinkInput = ({ label, description, value, locale }: { label: string, description: string, value: string, locale: string }) => {
    const { toast } = useToast();
    const [copied, setCopied] = useState(false);
    const t = translations[locale as keyof typeof translations] || translations.en;

    const handleCopy = () => {
        navigator.clipboard.writeText(value);
        setCopied(true);
        toast({ title: t.copied });
        setTimeout(() => setCopied(false), 2000);
    };

    return (
        <div className="space-y-2">
            <Label htmlFor={label}>{label}</Label>
            <div className="flex gap-2">
                <Input id={label} value={value} readOnly className="text-muted-foreground" />
                <Button variant="outline" size="icon" onClick={handleCopy}>
                    {copied ? <CheckCircle className="h-4 w-4 text-emerald-500" /> : <Copy className="h-4 w-4" />}
                </Button>
            </div>
            <p className="text-xs text-muted-foreground">{description}</p>
        </div>
    );
}

export function AgentCreatedDialog({ open, onOpenChange, agentId, locale }: AgentCreatedDialogProps) {
    const t = translations[locale as keyof typeof translations] || translations.en;
    const checkoutUrl = `https://app.agenmax.com/buy/${agentId}`;
    const embedCode = `<iframe src="${checkoutUrl}" width="100%" height="600px" frameborder="0"></iframe>`;
    const marketplaceUrl = `/${locale}/marketplace/${agentId}`;

    return (
        <Dialog open={open} onOpenChange={onOpenChange}>
            <DialogContent className="sm:max-w-md">
                <DialogHeader>
                    <div className="flex justify-center mb-4">
                        <CheckCircle className="h-16 w-16 text-green-500" />
                    </div>
                    <DialogTitle className="text-center text-2xl font-headline">{t.title}</DialogTitle>
                    <DialogDescription className="text-center">
                        {t.description}
                    </DialogDescription>
                </DialogHeader>
                <div className="space-y-4 py-4">
                   <LinkInput label={t.checkoutLink} description={t.checkoutDescription} value={checkoutUrl} locale={locale} />
                   <LinkInput label={t.embedCode} description={t.embedDescription} value={embedCode} locale={locale} />
                   
                    <div className="space-y-2">
                        <Label>{t.marketplaceLink}</Label>
                        <p className="text-xs text-muted-foreground">{t.marketplaceDescription}</p>
                         <Button variant="outline" className="w-full" asChild>
                            <Link href={marketplaceUrl} target="_blank">
                               {t.viewOnMarketplace} <ExternalLink className="ml-2 h-4 w-4"/>
                            </Link>
                        </Button>
                    </div>

                </div>
                 <DialogFooter className="sm:justify-center">
                    <Button type="button" onClick={() => onOpenChange(false)}>
                       {t.close}
                    </Button>
                </DialogFooter>
            </DialogContent>
        </Dialog>
    );
}
