
"use client";

import { useState } from "react";
import { type UseFormReturn } from "react-hook-form";
import { z } from "zod";
import { useToast } from "@/hooks/use-toast";

import { Button } from "@/components/ui/button";
import {
  Form,
  FormControl,
  FormDescription,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Card, CardContent, CardHeader, CardTitle, CardDescription as CardDescriptionComponent } from "@/components/ui/card";
import { Sparkles, Loader2, Settings, CreditCard, Percent, Share2, UploadCloud, Image as ImageIcon, Video, UserPlus } from "lucide-react";
import { Switch } from "@/components/ui/switch";
import { RadioGroup, RadioGroupItem } from "@/components/ui/radio-group";

export const formSchema = z.object({
  agentName: z.string().min(3, "El nombre del agente debe tener al menos 3 caracteres."),
  agentDescription: z.string().min(10, "La descripción debe tener al menos 10 caracteres."),
  bannerImage: z.any().optional(),
  customInstructions: z.string().optional(),
  inputChannels: z.object({
    chatEmbedding: z.boolean().optional(),
    whatsapp: z.boolean().optional(),
    websiteUrl: z.boolean().optional(),
    googleDrive: z.boolean().optional(),
    googleSheets: z.boolean().optional(),
    json: z.boolean().optional(),
  }),
  processingChannels: z.object({
      make: z.boolean().optional(),
      zapier: z.boolean().optional(),
      n8n: z.boolean().optional(),
      chatgpt: z.boolean().optional(),
      gemini: z.boolean().optional(),
      runway: z.boolean().optional(),
      midjourney: z.boolean().optional(),
      claude: z.boolean().optional(),
  }),
  outputChannels: z.object({
    chatEmbedding: z.boolean().optional(),
    whatsapp: z.boolean().optional(),
    websiteUrl: z.boolean().optional(),
    googleDrive: z.boolean().optional(),
    googleSheets: z.boolean().optional(),
    json: z.boolean().optional(),
  }),
  marketplaceDescription: z.string().optional(),
  marketplaceImages: z.any().optional(),
  marketplaceVideos: z.any().optional(),
  pricingModel: z.enum(["lifetime", "subscription"], {
    required_error: "Debes seleccionar un modelo de precios.",
  }),
  price: z.coerce.number().min(0, "El precio debe ser un número positivo.").optional(),
  affiliateEnabled: z.boolean().optional(),
  affiliateCommission: z.coerce.number().min(0).max(100).optional(),
});

interface CreateAgentFormProps {
    form: UseFormReturn<z.infer<typeof formSchema>>;
    step: number;
}

export function CreateAgentForm({ form, step }: CreateAgentFormProps) {
  const { toast } = useToast();
  const affiliateEnabled = form.watch("affiliateEnabled");

  return (
    <>
      {step === 1 && (
        <div className="space-y-8">
          <Card>
            <CardHeader>
              <CardTitle className="font-headline text-2xl">Identidad del Agente</CardTitle>
              <CardDescriptionComponent>Proporciona la información básica de tu agente.</CardDescriptionComponent>
            </CardHeader>
            <CardContent className="space-y-6">
              <FormField
                control={form.control}
                name="agentName"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Nombre del Agente</FormLabel>
                    <FormControl>
                      <Input placeholder="ej: Bot Procesador de Datos" {...field} />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <FormField
                control={form.control}
                name="agentDescription"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Descripción del Agente</FormLabel>
                    <FormControl>
                      <Textarea
                        placeholder="ej: Una IA que recibe datos de una hoja de cálculo, los procesa con Gemini y los guarda en Drive."
                        {...field}
                        rows={4}
                      />
                    </FormControl>
                    <FormDescription>
                      Describe en detalle qué hace tu agente.
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />
               <FormField
                  control={form.control}
                  name="bannerImage"
                  render={({ field }) => (
                      <FormItem>
                          <FormLabel>Banner del Agente</FormLabel>
                          <FormDescription>Sube una imagen de 600x400px para la tarjeta de tu agente en el marketplace.</FormDescription>
                          <FormControl>
                              <div className="flex flex-col items-center justify-center p-8 border-2 border-dashed rounded-lg text-center cursor-pointer hover:bg-accent/50 transition-colors">
                                  <UploadCloud className="h-12 w-12 text-muted-foreground"/>
                                  <p className="mt-4 text-sm font-semibold text-foreground">Haz clic para subir o arrastra y suelta</p>
                                  <p className="text-xs text-muted-foreground mt-1">PNG, JPG o GIF (Recom. 600x400px)</p>
                                  <Input type="file" className="sr-only" {...field} />
                              </div>
                          </FormControl>
                          <FormMessage />
                      </FormItem>
                  )}
              />
            </CardContent>
          </Card>
        </div>
      )}

       {step === 2 && (
            <div className="space-y-8">
                <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl">Canales de Entrada</CardTitle>
                        <CardDescriptionComponent>Selecciona cómo tu agente recibirá información.</CardDescriptionComponent>
                    </CardHeader>
                    <CardContent className="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <FormField control={form.control} name="inputChannels.chatEmbedding" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base">Chat Embebido</FormLabel><FormDescription>Permite que el agente interactúe en un sitio web.</FormDescription></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="inputChannels.whatsapp" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base">WhatsApp</FormLabel><FormDescription>Conecta el agente a un número de WhatsApp.</FormDescription></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="inputChannels.websiteUrl" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base">URL del Sitio Web</FormLabel><FormDescription>Usa el contenido de un sitio web como conocimiento.</FormDescription></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="inputChannels.googleDrive" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base">Google Drive</FormLabel><FormDescription>Importa documentos desde Google Drive.</FormDescription></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="inputChannels.googleSheets" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base">Google Sheets</FormLabel><FormDescription>Importa datos desde una hoja de cálculo de Google.</FormDescription></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="inputChannels.json" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base">Datos JSON</FormLabel><FormDescription>Proporciona datos a través de una estructura JSON.</FormDescription></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                    </CardContent>
                </Card>
                 <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl">Canales de Tratamiento</CardTitle>
                        <CardDescriptionComponent>Selecciona la herramienta externa que procesará la información.</CardDescriptionComponent>
                    </CardHeader>
                    <CardContent className="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <FormField control={form.control} name="processingChannels.make" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base flex items-center gap-2"><Share2 className="h-4 w-4"/>Make</FormLabel></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="processingChannels.zapier" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base flex items-center gap-2"><Share2 className="h-4 w-4"/>Zapier</FormLabel></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="processingChannels.n8n" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base flex items-center gap-2"><Share2 className="h-4 w-4"/>n8n</FormLabel></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="processingChannels.chatgpt" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base flex items-center gap-2"><Sparkles className="h-4 w-4"/>ChatGPT</FormLabel></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="processingChannels.gemini" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base flex items-center gap-2"><Sparkles className="h-4 w-4"/>Gemini</FormLabel></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="processingChannels.claude" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base flex items-center gap-2"><Sparkles className="h-4 w-4"/>Claude</FormLabel></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="processingChannels.midjourney" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base flex items-center gap-2"><Sparkles className="h-4 w-4"/>Midjourney</FormLabel></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="processingChannels.runway" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base flex items-center gap-2"><Sparkles className="h-4 w-4"/>Runway</FormLabel></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                    </CardContent>
                </Card>
                 <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl">Canales de Salida</CardTitle>
                        <CardDescriptionComponent>Selecciona cómo tu agente enviará la información.</CardDescriptionComponent>
                    </CardHeader>
                     <CardContent className="grid grid-cols-1 md:grid-cols-2 gap-6">
                       <FormField control={form.control} name="outputChannels.chatEmbedding" render={({field}) => (
                           <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                               <div className="space-y-0.5"><FormLabel className="text-base">Chat Embebido</FormLabel><FormDescription>Envía respuestas a un chat de sitio web.</FormDescription></div>
                               <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                           </FormItem>
                       )}/>
                        <FormField control={form.control} name="outputChannels.whatsapp" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base">WhatsApp</FormLabel><FormDescription>Envía mensajes a un número de WhatsApp.</FormDescription></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="outputChannels.websiteUrl" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base">URL del Sitio Web (Webhook)</FormLabel><FormDescription>Envía datos a una URL externa.</FormDescription></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="outputChannels.googleDrive" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base">Google Drive</FormLabel><FormDescription>Guarda documentos y archivos en Drive.</FormDescription></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="outputChannels.googleSheets" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base">Google Sheets</FormLabel><FormDescription>Exporta datos a una hoja de cálculo de Google.</FormDescription></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="outputChannels.json" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base">Datos JSON</FormLabel><FormDescription>Devuelve datos en una estructura JSON.</FormDescription></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                    </CardContent>
                </Card>
            </div>
        )}
        {step === 3 && (
            <div className="space-y-8">
                <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl">Página del Marketplace</CardTitle>
                        <CardDescriptionComponent>Configura cómo se mostrará tu agente a los clientes.</CardDescriptionComponent>
                    </CardHeader>
                    <CardContent className="space-y-6">
                        <FormField
                        control={form.control}
                        name="marketplaceDescription"
                        render={({ field }) => (
                            <FormItem>
                                <FormLabel>Descripción Detallada</FormLabel>
                                <FormControl>
                                    <div className="w-full rounded-md border border-input bg-background p-3">
                                        {/* This would be a rich text editor */}
                                        <Textarea
                                            placeholder="Describe tu agente en detalle. Usa formato para destacar características y beneficios..."
                                            className="min-h-[200px] border-0 p-0 focus-visible:ring-0 focus-visible:ring-offset-0"
                                            {...field}
                                        />
                                    </div>
                                </FormControl>
                                <FormDescription>
                                Este contenido se usará en la página de ventas de tu agente.
                                </FormDescription>
                                <FormMessage />
                            </FormItem>
                        )}
                        />
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                             <FormField
                                control={form.control}
                                name="marketplaceImages"
                                render={({ field }) => (
                                    <FormItem>
                                        <FormLabel>Imágenes Promocionales</FormLabel>
                                        <FormControl>
                                            <div className="flex flex-col items-center justify-center p-8 border-2 border-dashed rounded-lg text-center cursor-pointer hover:bg-accent/50 transition-colors">
                                                <ImageIcon className="h-12 w-12 text-muted-foreground"/>
                                                <p className="mt-4 text-sm font-semibold text-foreground">Haz clic para subir imágenes</p>
                                                <p className="text-xs text-muted-foreground mt-1">PNG, JPG, GIF</p>
                                                <Input type="file" multiple className="sr-only" {...field} />
                                            </div>
                                        </FormControl>
                                        <FormMessage />
                                    </FormItem>
                                )}
                            />
                             <FormField
                                control={form.control}
                                name="marketplaceVideos"
                                render={({ field }) => (
                                    <FormItem>
                                        <FormLabel>Video de Demostración</FormLabel>
                                        <FormControl>
                                            <div className="flex flex-col items-center justify-center p-8 border-2 border-dashed rounded-lg text-center cursor-pointer hover:bg-accent/50 transition-colors">
                                                <Video className="h-12 w-12 text-muted-foreground"/>
                                                <p className="mt-4 text-sm font-semibold text-foreground">Haz clic para subir videos</p>
                                                <p className="text-xs text-muted-foreground mt-1">MP4, WEBM</p>
                                                <Input type="file" multiple className="sr-only" {...field} />
                                            </div>
                                        </FormControl>
                                        <FormMessage />
                                    </FormItem>
                                )}
                            />
                        </div>
                    </CardContent>
                </Card>
                 <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl flex items-center gap-2"><UserPlus /> Programa de Afiliados</CardTitle>
                        <CardDescriptionComponent>
                           Permite que otros vendan tu agente a cambio de una comisión.
                        </CardDescriptionComponent>
                    </CardHeader>
                    <CardContent className="space-y-6">
                        <FormField
                            control={form.control}
                            name="affiliateEnabled"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">Activar Afiliación</FormLabel>
                                        <FormDescription>Permitir que otros promocionen tu agente.</FormDescription>
                                    </div>
                                    <FormControl>
                                        <Switch checked={field.value} onCheckedChange={field.onChange} />
                                    </FormControl>
                                </FormItem>
                            )}
                        />
                        {affiliateEnabled && (
                             <FormField
                                control={form.control}
                                name="affiliateCommission"
                                render={({ field }) => (
                                    <FormItem>
                                    <FormLabel>Comisión del Afiliado</FormLabel>
                                    <div className="relative">
                                        <FormControl>
                                            <Input type="number" placeholder="20" className="pr-8" {...field} />
                                        </FormControl>
                                        <span className="absolute inset-y-0 right-0 flex items-center pr-3 text-muted-foreground">%</span>
                                    </div>
                                    <FormDescription>El porcentaje que los afiliados recibirán por cada venta.</FormDescription>
                                    <FormMessage />
                                    </FormItem>
                                )}
                            />
                        )}
                    </CardContent>
                </Card>
            </div>
        )}
         {step === 4 && (
             <div className="space-y-8">
                <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl">Precios</CardTitle>
                        <CardDescriptionComponent>Establece un precio para tu agente.</CardDescriptionComponent>
                    </CardHeader>
                    <CardContent className="space-y-6">
                        <FormField
                        control={form.control}
                        name="pricingModel"
                        render={({ field }) => (
                            <FormItem className="space-y-3">
                            <FormLabel>Modelo de Precios</FormLabel>
                            <FormControl>
                                <RadioGroup
                                onValueChange={field.onChange}
                                defaultValue={field.value}
                                className="flex flex-col space-y-1"
                                >
                                <FormItem className="flex items-center space-x-3 space-y-0">
                                    <FormControl>
                                    <RadioGroupItem value="lifetime" />
                                    </FormControl>
                                    <FormLabel className="font-normal">
                                    Acceso Vitalicio (pago único)
                                    </FormLabel>
                                </FormItem>
                                <FormItem className="flex items-center space-x-3 space-y-0">
                                    <FormControl>
                                    <RadioGroupItem value="subscription" />
                                    </FormControl>
                                    <FormLabel className="font-normal">
                                     Suscripción (recurrente mensual)
                                    </FormLabel>
                                </FormItem>
                                </RadioGroup>
                            </FormControl>
                            <FormMessage />
                            </FormItem>
                        )}
                        />
                         <FormField
                            control={form.control}
                            name="price"
                            render={({ field }) => (
                                <FormItem>
                                <FormLabel>Precio</FormLabel>
                                <div className="relative">
                                    <span className="absolute inset-y-0 left-0 flex items-center pl-3 text-muted-foreground">$</span>
                                    <FormControl>
                                        <Input type="number" placeholder="49.99" className="pl-7" {...field} />
                                    </FormControl>
                                     <span className="absolute inset-y-0 right-0 flex items-center pr-3 text-muted-foreground">
                                        {form.getValues("pricingModel") === "subscription" ? "/ mes" : ""}
                                    </span>
                                </div>
                                <FormMessage />
                                </FormItem>
                            )}
                        />
                    </CardContent>
                </Card>
                 <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl flex items-center gap-2"><CreditCard /> Pagos</CardTitle>
                        <CardDescriptionComponent>
                            Los pagos se procesan a través de Stripe.
                        </CardDescriptionComponent>
                    </CardHeader>
                    <CardContent className="space-y-4">
                        <div className="flex items-center justify-between rounded-lg border p-4">
                            <div>
                                <p className="font-semibold">Stripe Connect</p>
                                <p className="text-sm text-muted-foreground">Conectado: <span className="text-primary font-medium">acct_...****</span></p>
                            </div>
                            <Button variant="outline">Gestionar</Button>
                        </div>
                        <div className="flex items-center justify-between rounded-lg border p-4 bg-muted/20">
                           <div className="flex items-center gap-2 text-sm text-muted-foreground">
                             <Percent className="h-4 w-4" />
                             <p>Comisión de la plataforma por transacción</p>
                           </div>
                           <p className="text-sm font-semibold">10%</p>
                        </div>
                    </CardContent>
                </Card>
            </div>
        )}
    </>
  );
}
