
"use client";

import { useState } from "react";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { AppLayout } from "@/components/layout/AppLayout";
import { CreateAgentForm, formSchema } from "./CreateAgentForm";
import { AgentTestChat } from "./AgentTestChat";
import { Button } from "@/components/ui/button";
import { Bot, ArrowRight, ArrowLeft } from "lucide-react";
import { Form } from "@/components/ui/form";
import { Progress } from "@/components/ui/progress";
import { AgentCreatedDialog } from "@/components/agent/AgentCreatedDialog";

export default function CreateAgentPage() {
    const [step, setStep] = useState(1);
    const [isDialogOpen, setIsDialogOpen] = useState(false);

    const totalSteps = 5;

    const form = useForm<z.infer<typeof formSchema>>({
        resolver: zodResolver(formSchema),
        defaultValues: {
            agentName: "",
            agentDescription: "",
            persona: "",
            toneOfVoice: "",
            knowledgeBaseText: "",
            knowledgeBaseUrl: "",
            customInstructions: "",
            inputChannels: {
                chatEmbedding: false,
                whatsapp: false,
                websiteUrl: false,
                googleDrive: false,
                googleSheets: false,
                json: false,
            },
            outputChannels: {
                chatEmbedding: false,
                whatsapp: false,
                websiteUrl: false,
                googleDrive: false,
                googleSheets: false,
                json: false,
            },
            marketplaceDescription: "",
            pricingModel: "lifetime",
            price: undefined,
            affiliateEnabled: false,
            affiliateCommission: 20,
            supportLink: "",
            termsOfUseLink: "",
        },
        mode: "onChange",
    });

    const agentName = form.watch("agentName");
    const persona = form.watch("persona");

    const nextStep = () => setStep((prev) => prev < totalSteps ? prev + 1 : prev);
    const prevStep = () => setStep((prev) => prev > 1 ? prev - 1 : prev);

    function onSubmit(values: z.infer<typeof formSchema>) {
        console.log(values);
        setIsDialogOpen(true);
    }

    const translations = {
        dashboard: "Dashboard",
        myAgents: "Mis Agentes",
        sales: "Ventas",
        purchases: "Compras",
        myAccount: "Mi Cuenta",
        support: "Soporte",
        marketplace: "Marketplace",
        createAgent: "Crear Agente",
        settings: "Configuración",
        logout: "Cerrar Sesión",
        affiliates: "Afiliados"
    };

    return (
        <AppLayout translations={translations}>
            <div className="flex items-center justify-between mb-4">
                <div className="flex items-center gap-4">
                    <div className="p-3 bg-primary/10 rounded-lg">
                        <Bot className="h-8 w-8 text-primary" />
                    </div>
                    <div>
                        <h1 className="text-4xl font-bold font-headline tracking-tight">
                            Crea Tu Agente de IA
                        </h1>
                        <p className="text-muted-foreground mt-1">
                            Sigue los pasos para dar vida a tu agente.
                        </p>
                    </div>
                </div>
            </div>

            <div className="my-8 space-y-4">
                <Progress value={(step / totalSteps) * 100} className="h-2" />
                <div className="flex justify-between mt-2 text-sm font-medium text-muted-foreground">
                    <span className={step >= 1 ? 'text-primary font-semibold' : ''}>Identidad</span>
                    <span className={step >= 2 ? 'text-primary font-semibold' : ''}>Conocimiento</span>
                    <span className={step >= 3 ? 'text-primary font-semibold' : ''}>Canales</span>
                    <span className={step >= 4 ? 'text-primary font-semibold' : ''}>Marketplace</span>
                    <span className={step >= 5 ? 'text-primary font-semibold' : ''}>Precios</span>
                </div>
            </div>

            <div className="grid grid-cols-1 lg:grid-cols-3 gap-12 mt-8">
                <div className="lg:col-span-2">
                    <Form {...form}>
                        <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-8">
                            <CreateAgentForm form={form} step={step} />
                            
                            <div className="flex justify-between items-center gap-4 pt-4 border-t">
                                <div>
                                    {step > 1 && (
                                        <Button type="button" variant="outline" size="lg" onClick={prevStep}>
                                            <ArrowLeft className="mr-2 h-4 w-4" />
                                            Anterior
                                        </Button>
                                    )}
                                </div>
                                <div className="flex items-center gap-4">
                                    <Button type="button" variant="ghost" size="lg">Guardar como Borrador</Button>
                                    {step < totalSteps ? (
                                        <Button type="button" size="lg" onClick={nextStep}>
                                            Siguiente
                                            <ArrowRight className="ml-2 h-4 w-4" />
                                        </Button>
                                    ) : (
                                        <Button type="submit" size="lg">
                                            <Bot className="mr-2 h-5 w-5" />
                                            Crear Agente
                                        </Button>
                                    )}
                                </div>
                            </div>
                        </form>
                    </Form>
                </div>
                <div className="max-w-full">
                    <div className="sticky top-24">
                        <AgentTestChat agentName={agentName} persona={persona} />
                    </div>
                </div>
            </div>
            <AgentCreatedDialog 
                open={isDialogOpen} 
                onOpenChange={setIsDialogOpen}
                agentId="agent-123"
                locale="es"
            />
        </AppLayout>
    );
}

    