
"use client";

import { useState } from "react";
import { type UseFormReturn } from "react-hook-form";
import { z } from "zod";
import { generateAgentPersona } from "@/ai/flows/agent-persona-generator";
import { generateKnowledgeBase } from "@/ai/flows/knowledge-base-generator";
import { useToast } from "@/hooks/use-toast";

import { Button } from "@/components/ui/button";
import {
  Form,
  FormControl,
  FormDescription,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Card, CardContent, CardHeader, CardTitle, CardDescription as CardDescriptionComponent } from "@/components/ui/card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Sparkles, UploadCloud, Link as LinkIcon, FileText, Loader2, Settings, CreditCard, Percent, Image as ImageIcon, Video, UserPlus, FileQuestion, ShieldCheck } from "lucide-react";
import { Switch } from "@/components/ui/switch";
import { RadioGroup, RadioGroupItem } from "@/components/ui/radio-group";

export const formSchema = z.object({
  agentName: z.string().min(3, "El nombre del agente debe tener al menos 3 caracteres."),
  agentDescription: z.string().min(10, "La descripción debe tener al menos 10 caracteres."),
  bannerImage: z.any().optional(),
  persona: z.string().min(10, "La persona debe tener al menos 10 caracteres."),
  toneOfVoice: z.string().min(5, "El tono de voz debe tener al menos 5 caracteres."),
  knowledgeBaseText: z.string().optional(),
  knowledgeBaseUrl: z.string().url().optional().or(z.literal('')),
  customInstructions: z.string().optional(),
  inputChannels: z.object({
    chatEmbedding: z.boolean().optional(),
    whatsapp: z.boolean().optional(),
    websiteUrl: z.boolean().optional(),
    googleDrive: z.boolean().optional(),
    googleSheets: z.boolean().optional(),
    json: z.boolean().optional(),
  }),
  outputChannels: z.object({
    chatEmbedding: z.boolean().optional(),
    whatsapp: z.boolean().optional(),
    websiteUrl: z.boolean().optional(),
    googleDrive: z.boolean().optional(),
    googleSheets: z.boolean().optional(),
    json: z.boolean().optional(),
  }),
  marketplaceDescription: z.string().optional(),
  marketplaceImages: z.any().optional(),
  marketplaceVideos: z.any().optional(),
  supportLink: z.string().url().optional().or(z.literal('')),
  termsOfUseLink: z.string().url().optional().or(z.literal('')),
  pricingModel: z.enum(["lifetime", "subscription"], {
    required_error: "Debes seleccionar un modelo de precios.",
  }),
  price: z.coerce.number().min(0, "El precio debe ser un número positivo.").optional(),
  affiliateEnabled: z.boolean().optional(),
  affiliateCommission: z.coerce.number().min(0).max(100).optional(),
});

interface CreateAgentFormProps {
    form: UseFormReturn<z.infer<typeof formSchema>>;
    step: number;
}

export function CreateAgentForm({ form, step }: CreateAgentFormProps) {
  const [isGeneratingPersona, setIsGeneratingPersona] = useState(false);
  const [isGeneratingKnowledge, setIsGeneratingKnowledge] = useState(false);
  const { toast } = useToast();
  const affiliateEnabled = form.watch("affiliateEnabled");

  const handleGeneratePersona = async () => {
    const agentDescription = form.getValues("agentDescription");
    if (!agentDescription) {
      form.setError("agentDescription", {
        type: "manual",
        message: "Por favor, proporciona una descripción para generar una persona.",
      });
      return;
    }

    setIsGeneratingPersona(true);
    try {
      const result = await generateAgentPersona({ agentDescription });
      form.setValue("persona", result.persona, { shouldValidate: true });
      form.setValue("toneOfVoice", result.toneOfVoice, { shouldValidate: true });
      toast({
        title: "¡Personalidad Generada!",
        description: "La persona y el tono de voz de tu agente han sido creados.",
      });
    } catch (error) {
      console.error("Fallo al generar persona:", error);
      toast({
        variant: "destructive",
        title: "Fallo en la Generación",
        description: "No se pudo generar la personalidad. Por favor, inténtalo de nuevo.",
      });
    } finally {
      setIsGeneratingPersona(false);
    }
  };

  const handleGenerateKnowledge = async () => {
    const agentDescription = form.getValues("agentDescription");
     if (!agentDescription) {
      form.setError("agentDescription", {
        type: "manual",
        message: "Por favor, proporciona una descripción para generar la base de conocimiento.",
      });
      return;
    }
    setIsGeneratingKnowledge(true);
    try {
      const result = await generateKnowledgeBase({ agentDescription });
      form.setValue("knowledgeBaseText", result.knowledgeBaseText, { shouldValidate: true });
      toast({
        title: "¡Base de Conocimiento Generada!",
        description: "Se ha creado un texto de inicio basado en la descripción de tu agente.",
      });
    } catch (error) {
        console.error("Fallo al generar conocimiento:", error);
        toast({
            variant: "destructive",
            title: "Fallo en la Generación",
            description: "No se pudo generar la base de conocimiento. Por favor, inténtalo de nuevo.",
        });
    } finally {
        setIsGeneratingKnowledge(false);
    }
  }

  return (
    <>
      {step === 1 && (
        <div className="space-y-8">
          <Card>
            <CardHeader>
              <CardTitle className="font-headline text-2xl">Identidad del Agente</CardTitle>
              <CardDescriptionComponent>Proporciona la información básica de tu agente.</CardDescriptionComponent>
            </CardHeader>
            <CardContent className="space-y-6">
              <FormField
                control={form.control}
                name="agentName"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Nombre del Agente</FormLabel>
                    <FormControl>
                      <Input placeholder="ej: Gurú de Marketing" {...field} />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <FormField
                control={form.control}
                name="agentDescription"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Descripción del Agente</FormLabel>
                    <FormControl>
                      <Textarea
                        placeholder="ej: Una IA que genera textos creativos para marketing, analiza tendencias y sugiere campañas..."
                        {...field}
                        rows={4}
                      />
                    </FormControl>
                    <FormDescription>
                      Describe en detalle qué hace tu agente. Esto es crucial para generar la personalidad y el conocimiento.
                    </FormDescription>
                    <FormMessage />
                  </FormItem>
                )}
              />
               <FormField
                  control={form.control}
                  name="bannerImage"
                  render={({ field }) => (
                      <FormItem>
                          <FormLabel>Banner del Agente</FormLabel>
                          <FormDescription>Sube una imagen de 600x400px para la tarjeta de tu agente en el marketplace.</FormDescription>
                          <FormControl>
                              <div className="flex flex-col items-center justify-center p-8 border-2 border-dashed rounded-lg text-center cursor-pointer hover:bg-accent/50 transition-colors">
                                  <UploadCloud className="h-12 w-12 text-muted-foreground"/>
                                  <p className="mt-4 text-sm font-semibold text-foreground">Haz clic para subir o arrastra y suelta</p>
                                  <p className="text-xs text-muted-foreground mt-1">PNG, JPG o GIF (Recom. 600x400px)</p>
                                  <Input type="file" className="sr-only" {...field} />
                              </div>
                          </FormControl>
                          <FormMessage />
                      </FormItem>
                  )}
              />
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <div>
                <CardTitle className="font-headline text-2xl">Personalidad</CardTitle>
                <CardDescriptionComponent>Define la persona y el tono de voz de tu agente, o genéralo con IA.</CardDescriptionComponent>
              </div>
            </CardHeader>
            <CardContent className="space-y-6">
              <FormField
                control={form.control}
                name="persona"
                render={({ field }) => (
                  <FormItem>
                     <div className="flex justify-between items-center">
                        <FormLabel>Persona</FormLabel>
                        <Button type="button" variant="ghost" size="sm" onClick={handleGeneratePersona} disabled={isGeneratingPersona}>
                            {isGeneratingPersona ? <Loader2 className="mr-2 h-4 w-4 animate-spin" /> : <Sparkles className="mr-2 h-4 w-4" />}
                            Generar
                        </Button>
                     </div>
                    <FormControl>
                      <Textarea
                        placeholder="Describe la personalidad de tu agente. ¿Es formal, ingenioso, empático?"
                        className="min-h-[120px]"
                        {...field}
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <FormField
                control={form.control}
                name="toneOfVoice"
                render={({ field }) => (
                  <FormItem>
                    <div className="flex justify-between items-center">
                        <FormLabel>Tono de Voz</FormLabel>
                        <Button type="button" variant="ghost" size="sm" onClick={handleGeneratePersona} disabled={isGeneratingPersona}>
                            {isGeneratingPersona ? <Loader2 className="mr-2 h-4 w-4 animate-spin" /> : <Sparkles className="mr-2 h-4 w-4" />}
                            Generar
                        </Button>
                    </div>
                    <FormControl>
                      <Textarea
                        placeholder="Define el estilo de hablar del agente. Usa frases cortas, emojis, tono formal, etc."
                        {...field}
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </CardContent>
          </Card>
        </div>
      )}

      {step === 2 && (
        <div className="space-y-8">
          <Card>
            <CardHeader>
              <div>
                <CardTitle className="font-headline text-2xl">Base de Conocimiento</CardTitle>
                <CardDescriptionComponent>
                  Proporciona la información que tu agente usará para responder. Puedes pegar, generar con IA, subir o importar desde un enlace.
                </CardDescriptionComponent>
              </div>
            </CardHeader>
            <CardContent>
              <Tabs defaultValue="text" className="w-full">
                <TabsList className="grid w-full grid-cols-3">
                  <TabsTrigger value="text"><FileText className="mr-2 h-4 w-4" />Texto</TabsTrigger>
                  <TabsTrigger value="upload"><UploadCloud className="mr-2 h-4 w-4" />Subir PDF</TabsTrigger>
                  <TabsTrigger value="link"><LinkIcon className="mr-2 h-4 w-4" />Enlace</TabsTrigger>
                </TabsList>
                <TabsContent value="text" className="mt-4">
                  <FormField
                    control={form.control}
                    name="knowledgeBaseText"
                    render={({ field }) => (
                      <FormItem>
                         <div className="flex justify-between items-center mb-2">
                            <FormLabel>Pegar Texto</FormLabel>
                            <Button type="button" variant="ghost" size="sm" onClick={handleGenerateKnowledge} disabled={isGeneratingKnowledge}>
                                {isGeneratingKnowledge ? <Loader2 className="mr-2 h-4 w-4 animate-spin" /> : <Sparkles className="mr-2 h-4 w-4" />}
                                {isGeneratingKnowledge ? "Generando..." : "Generar con IA"}
                            </Button>
                        </div>
                        <FormControl>
                          <Textarea
                            placeholder="Pega el contenido de tu base de conocimiento aquí o genera con IA."
                            className="min-h-[250px]"
                            {...field}
                          />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </TabsContent>
                <TabsContent value="upload" className="mt-4">
                  <div className="flex flex-col items-center justify-center p-10 border-2 border-dashed rounded-lg text-center">
                    <UploadCloud className="h-12 w-12 text-muted-foreground" />
                    <p className="mt-4 text-sm font-semibold text-foreground">Arrastra y suelta un archivo PDF</p>
                    <p className="text-xs text-muted-foreground mt-1">o haz clic para seleccionar</p>
                    <Button type="button" variant="outline" className="mt-4">Seleccionar Archivo</Button>
                  </div>
                </TabsContent>
                <TabsContent value="link" className="mt-4">
                  <FormField
                    control={form.control}
                    name="knowledgeBaseUrl"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Importar desde URL</FormLabel>
                        <FormControl>
                          <Input placeholder="https://ejemplo.com/conocimiento" {...field} />
                        </FormControl>
                        <FormDescription>
                          Extraeremos el contenido de esta página.
                        </FormDescription>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </TabsContent>
              </Tabs>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle className="font-headline text-2xl">Instrucciones Personalizadas</CardTitle>
              <CardDescriptionComponent>Proporciona reglas y directrices específicas para el comportamiento de tu agente.</CardDescriptionComponent>
            </CardHeader>
            <CardContent>
              <FormField
                control={form.control}
                name="customInstructions"
                render={({ field }) => (
                  <FormItem>
                    <FormControl>
                      <Textarea
                        placeholder="Ej: Siempre responde en Español. Nunca menciones a la competencia. Siempre incluye una llamada a la acción a nuestro sitio web."
                        className="min-h-[150px]"
                        {...field}
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </CardContent>
          </Card>
        </div>
      )}
       {step === 3 && (
            <div className="space-y-8">
                <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl">Canales de Entrada</CardTitle>
                        <CardDescriptionComponent>Selecciona cómo tu agente recibirá información.</CardDescriptionComponent>
                    </CardHeader>
                    <CardContent className="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <FormField
                            control={form.control}
                            name="inputChannels.chatEmbedding"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">Chat Embebido</FormLabel>
                                        <FormDescription>Permite que el agente interactúe en un sitio web.</FormDescription>
                                    </div>
                                     <div className="flex items-center gap-2">
                                        <FormControl>
                                            <Switch checked={field.value} onCheckedChange={field.onChange} />
                                        </FormControl>
                                        <Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button>
                                    </div>
                                </FormItem>
                            )}
                        />
                        <FormField
                            control={form.control}
                            name="inputChannels.whatsapp"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">WhatsApp</FormLabel>
                                        <FormDescription>Conecta el agente a un número de WhatsApp.</FormDescription>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <FormControl>
                                            <Switch checked={field.value} onCheckedChange={field.onChange} />
                                        </FormControl>
                                        <Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button>
                                    </div>
                                </FormItem>
                            )}
                        />
                         <FormField
                            control={form.control}
                            name="inputChannels.websiteUrl"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">URL del Sitio Web</FormLabel>
                                        <FormDescription>Usa el contenido de un sitio web como conocimiento.</FormDescription>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <FormControl>
                                            <Switch checked={field.value} onCheckedChange={field.onChange} />
                                        </FormControl>
                                        <Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button>
                                    </div>
                                </FormItem>
                            )}
                        />
                         <FormField
                            control={form.control}
                            name="inputChannels.googleDrive"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">Google Drive</FormLabel>
                                        <FormDescription>Importa documentos desde Google Drive.</FormDescription>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <FormControl>
                                            <Switch checked={field.value} onCheckedChange={field.onChange} />
                                        </FormControl>
                                        <Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button>
                                    </div>
                                </FormItem>
                            )}
                        />
                         <FormField
                            control={form.control}
                            name="inputChannels.googleSheets"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">Google Sheets</FormLabel>
                                        <FormDescription>Importa datos desde una hoja de cálculo de Google.</FormDescription>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <FormControl>
                                            <Switch checked={field.value} onCheckedChange={field.onChange} />
                                        </FormControl>
                                        <Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button>
                                    </div>
                                </FormItem>
                            )}
                        />
                         <FormField
                            control={form.control}
                            name="inputChannels.json"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">Datos JSON</FormLabel>
                                        <FormDescription>Proporciona datos a través de una estructura JSON.</FormDescription>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <FormControl>
                                            <Switch checked={field.value} onCheckedChange={field.onChange} />
                                        </FormControl>
                                        <Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button>
                                    </div>
                                </FormItem>
                            )}
                        />
                    </CardContent>
                </Card>
                 <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl">Canales de Salida</CardTitle>
                        <CardDescriptionComponent>Selecciona cómo tu agente enviará la información.</CardDescriptionComponent>
                    </CardHeader>
                     <CardContent className="grid grid-cols-1 md:grid-cols-2 gap-6">
                       <FormField
                            control={form.control}
                            name="outputChannels.chatEmbedding"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">Chat Embebido</FormLabel>
                                        <FormDescription>Envía respuestas a un chat de sitio web.</FormDescription>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <FormControl>
                                            <Switch checked={field.value} onCheckedChange={field.onChange} />
                                        </FormControl>
                                        <Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button>
                                    </div>
                                </FormItem>
                            )}
                        />
                         <FormField
                            control={form.control}
                            name="outputChannels.whatsapp"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">WhatsApp</FormLabel>
                                        <FormDescription>Envía mensajes a un número de WhatsApp.</FormDescription>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <FormControl>
                                            <Switch checked={field.value} onCheckedChange={field.onChange} />
                                        </FormControl>
                                        <Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button>
                                    </div>
                                </FormItem>
                            )}
                        />
                         <FormField
                            control={form.control}
                            name="outputChannels.websiteUrl"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">URL del Sitio Web (Webhook)</FormLabel>
                                        <FormDescription>Envía datos a una URL externa.</FormDescription>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <FormControl>
                                            <Switch checked={field.value} onCheckedChange={field.onChange} />
                                        </FormControl>
                                        <Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button>
                                    </div>
                                </FormItem>
                            )}
                        />
                         <FormField
                            control={form.control}
                            name="outputChannels.googleDrive"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">Google Drive</FormLabel>
                                        <FormDescription>Guarda documentos y archivos en Drive.</FormDescription>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <FormControl>
                                            <Switch checked={field.value} onCheckedChange={field.onChange} />
                                        </FormControl>
                                        <Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button>
                                    </div>
                                </FormItem>
                            )}
                        />
                         <FormField
                            control={form.control}
                            name="outputChannels.googleSheets"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">Google Sheets</FormLabel>
                                        <FormDescription>Exporta datos a una hoja de cálculo de Google.</FormDescription>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <FormControl>
                                            <Switch checked={field.value} onCheckedChange={field.onChange} />
                                        </FormControl>
                                        <Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button>
                                    </div>
                                </FormItem>
                            )}
                        />
                         <FormField
                            control={form.control}
                            name="outputChannels.json"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">Datos JSON</FormLabel>
                                        <FormDescription>Devuelve datos en una estructura JSON.</FormDescription>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <FormControl>
                                            <Switch checked={field.value} onCheckedChange={field.onChange} />
                                        </FormControl>
                                        <Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button>
                                    </div>
                                </FormItem>
                            )}
                        />
                    </CardContent>
                </Card>
            </div>
        )}
        {step === 4 && (
            <div className="space-y-8">
                <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl">Página del Marketplace</CardTitle>
                        <CardDescriptionComponent>Configura cómo se mostrará tu agente a los clientes.</CardDescriptionComponent>
                    </CardHeader>
                    <CardContent className="space-y-6">
                        <FormField
                        control={form.control}
                        name="marketplaceDescription"
                        render={({ field }) => (
                            <FormItem>
                                <FormLabel>Descripción Detallada</FormLabel>
                                <FormControl>
                                    <div className="w-full rounded-md border border-input bg-background p-3">
                                        {/* This would be a rich text editor */}
                                        <Textarea
                                            placeholder="Describe tu agente en detalle. Usa formato para destacar características y beneficios..."
                                            className="min-h-[200px] border-0 p-0 focus-visible:ring-0 focus-visible:ring-offset-0"
                                            {...field}
                                        />
                                    </div>
                                </FormControl>
                                <FormDescription>
                                Este contenido se usará en la página de ventas de tu agente.
                                </FormDescription>
                                <FormMessage />
                            </FormItem>
                        )}
                        />
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                             <FormField
                                control={form.control}
                                name="marketplaceImages"
                                render={({ field }) => (
                                    <FormItem>
                                        <FormLabel>Imágenes Promocionales</FormLabel>
                                        <FormControl>
                                            <div className="flex flex-col items-center justify-center p-8 border-2 border-dashed rounded-lg text-center cursor-pointer hover:bg-accent/50 transition-colors">
                                                <ImageIcon className="h-12 w-12 text-muted-foreground"/>
                                                <p className="mt-4 text-sm font-semibold text-foreground">Haz clic para subir imágenes</p>
                                                <p className="text-xs text-muted-foreground mt-1">PNG, JPG, GIF</p>
                                                <Input type="file" multiple className="sr-only" {...field} />
                                            </div>
                                        </FormControl>
                                        <FormMessage />
                                    </FormItem>
                                )}
                            />
                             <FormField
                                control={form.control}
                                name="marketplaceVideos"
                                render={({ field }) => (
                                    <FormItem>
                                        <FormLabel>Video de Demostración</FormLabel>
                                        <FormControl>
                                            <div className="flex flex-col items-center justify-center p-8 border-2 border-dashed rounded-lg text-center cursor-pointer hover:bg-accent/50 transition-colors">
                                                <Video className="h-12 w-12 text-muted-foreground"/>
                                                <p className="mt-4 text-sm font-semibold text-foreground">Haz clic para subir videos</p>
                                                <p className="text-xs text-muted-foreground mt-1">MP4, WEBM</p>
                                                <Input type="file" multiple className="sr-only" {...field} />
                                            </div>
                                        </FormControl>
                                        <FormMessage />
                                    </FormItem>
                                )}
                            />
                        </div>
                         <FormField
                            control={form.control}
                            name="supportLink"
                            render={({ field }) => (
                                <FormItem>
                                    <FormLabel>Enlace de Soporte</FormLabel>
                                    <div className="flex items-center gap-2">
                                        <FileQuestion className="h-5 w-5 text-muted-foreground" />
                                        <FormControl>
                                            <Input placeholder="https://ejemplo.com/soporte" {...field} />
                                        </FormControl>
                                    </div>
                                    <FormMessage />
                                </FormItem>
                            )}
                        />
                        <FormField
                            control={form.control}
                            name="termsOfUseLink"
                            render={({ field }) => (
                                <FormItem>
                                    <FormLabel>Enlace de Términos de Uso</FormLabel>
                                     <div className="flex items-center gap-2">
                                        <ShieldCheck className="h-5 w-5 text-muted-foreground" />
                                        <FormControl>
                                            <Input placeholder="https://ejemplo.com/terminos" {...field} />
                                        </FormControl>
                                    </div>
                                    <FormMessage />
                                </FormItem>
                            )}
                        />
                    </CardContent>
                </Card>
                 <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl flex items-center gap-2"><UserPlus /> Programa de Afiliados</CardTitle>
                        <CardDescriptionComponent>
                           Permite que otros vendan tu agente a cambio de una comisión.
                        </CardDescriptionComponent>
                    </CardHeader>
                    <CardContent className="space-y-6">
                        <FormField
                            control={form.control}
                            name="affiliateEnabled"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">Activar Afiliación</FormLabel>
                                        <FormDescription>Permitir que otros promocionen tu agente.</FormDescription>
                                    </div>
                                    <FormControl>
                                        <Switch checked={field.value} onCheckedChange={field.onChange} />
                                    </FormControl>
                                </FormItem>
                            )}
                        />
                        {affiliateEnabled && (
                             <FormField
                                control={form.control}
                                name="affiliateCommission"
                                render={({ field }) => (
                                    <FormItem>
                                    <FormLabel>Comisión del Afiliado</FormLabel>
                                    <div className="relative">
                                        <FormControl>
                                            <Input type="number" placeholder="20" className="pr-8" {...field} />
                                        </FormControl>
                                        <span className="absolute inset-y-0 right-0 flex items-center pr-3 text-muted-foreground">%</span>
                                    </div>
                                    <FormDescription>El porcentaje que los afiliados recibirán por cada venta.</FormDescription>
                                    <FormMessage />
                                    </FormItem>
                                )}
                            />
                        )}
                    </CardContent>
                </Card>
            </div>
        )}
         {step === 5 && (
             <div className="space-y-8">
                <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl">Precios</CardTitle>
                        <CardDescriptionComponent>Establece un precio para tu agente.</CardDescriptionComponent>
                    </CardHeader>
                    <CardContent className="space-y-6">
                        <FormField
                        control={form.control}
                        name="pricingModel"
                        render={({ field }) => (
                            <FormItem className="space-y-3">
                            <FormLabel>Modelo de Precios</FormLabel>
                            <FormControl>
                                <RadioGroup
                                onValueChange={field.onChange}
                                defaultValue={field.value}
                                className="flex flex-col space-y-1"
                                >
                                <FormItem className="flex items-center space-x-3 space-y-0">
                                    <FormControl>
                                    <RadioGroupItem value="lifetime" />
                                    </FormControl>
                                    <FormLabel className="font-normal">
                                    Acceso Vitalicio (pago único)
                                    </FormLabel>
                                </FormItem>
                                <FormItem className="flex items-center space-x-3 space-y-0">
                                    <FormControl>
                                    <RadioGroupItem value="subscription" />
                                    </FormControl>
                                    <FormLabel className="font-normal">
                                     Suscripción (recurrente mensual)
                                    </FormLabel>
                                </FormItem>
                                </RadioGroup>
                            </FormControl>
                            <FormMessage />
                            </FormItem>
                        )}
                        />
                         <FormField
                            control={form.control}
                            name="price"
                            render={({ field }) => (
                                <FormItem>
                                <FormLabel>Precio</FormLabel>
                                <div className="relative">
                                    <span className="absolute inset-y-0 left-0 flex items-center pl-3 text-muted-foreground">$</span>
                                    <FormControl>
                                        <Input type="number" placeholder="49.99" className="pl-7" {...field} />
                                    </FormControl>
                                     <span className="absolute inset-y-0 right-0 flex items-center pr-3 text-muted-foreground">
                                        {form.getValues("pricingModel") === "subscription" ? "/ mes" : ""}
                                    </span>
                                </div>
                                <FormMessage />
                                </FormItem>
                            )}
                        />
                    </CardContent>
                </Card>
                 <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl flex items-center gap-2"><CreditCard /> Pagos</CardTitle>
                        <CardDescriptionComponent>
                            Los pagos se procesan a través de Stripe.
                        </CardDescriptionComponent>
                    </CardHeader>
                    <CardContent className="space-y-4">
                        <div className="flex items-center justify-between rounded-lg border p-4">
                            <div>
                                <p className="font-semibold">Stripe Connect</p>
                                <p className="text-sm text-muted-foreground">Cuenta conectada: acct_...****</p>
                            </div>
                            <Button variant="outline">Gestionar</Button>
                        </div>
                        <div className="flex items-center justify-between rounded-lg border p-4 bg-muted/20">
                           <div className="flex items-center gap-2 text-sm text-muted-foreground">
                             <Percent className="h-4 w-4" />
                             <p>Comisión de la plataforma por transacción</p>
                           </div>
                           <p className="text-sm font-semibold">10%</p>
                        </div>
                    </CardContent>
                </Card>
            </div>
        )}
    </>
  );
}

    