
"use client";

import { useState } from "react";
import { type UseFormReturn } from "react-hook-form";
import { z } from "zod";
import { useToast } from "@/hooks/use-toast";

import { Button } from "@/components/ui/button";
import {
  Form,
  FormControl,
  FormDescription,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Card, CardContent, CardHeader, CardTitle, CardDescription as CardDescriptionComponent } from "@/components/ui/card";
import { Sparkles, Loader2, Settings, CreditCard, Percent, Share2, UploadCloud, Image as ImageIcon, Video, UserPlus } from "lucide-react";
import { Switch } from "@/components/ui/switch";
import { RadioGroup, RadioGroupItem } from "@/components/ui/radio-group";

export const formSchema = z.object({
  agentName: z.string().min(3, "Agent name must be at least 3 characters."),
  agentDescription: z.string().min(10, "Description must be at least 10 characters."),
  bannerImage: z.any().optional(),
  customInstructions: z.string().optional(),
  inputChannels: z.object({
    chatEmbedding: z.boolean().optional(),
    whatsapp: z.boolean().optional(),
    websiteUrl: z.boolean().optional(),
    googleDrive: z.boolean().optional(),
    googleSheets: z.boolean().optional(),
    json: z.boolean().optional(),
  }),
  processingChannels: z.object({
      make: z.boolean().optional(),
      zapier: z.boolean().optional(),
      n8n: z.boolean().optional(),
      chatgpt: z.boolean().optional(),
      gemini: z.boolean().optional(),
      runway: z.boolean().optional(),
      midjourney: z.boolean().optional(),
      claude: z.boolean().optional(),
  }),
  outputChannels: z.object({
    chatEmbedding: z.boolean().optional(),
    whatsapp: z.boolean().optional(),
    websiteUrl: z.boolean().optional(),
    googleDrive: z.boolean().optional(),
    googleSheets: z.boolean().optional(),
    json: z.boolean().optional(),
  }),
  marketplaceDescription: z.string().optional(),
  marketplaceImages: z.any().optional(),
  marketplaceVideos: z.any().optional(),
  pricingModel: z.enum(["lifetime", "subscription"], {
    required_error: "You need to select a pricing model.",
  }),
  price: z.coerce.number().min(0, "Price must be a positive number.").optional(),
  affiliateEnabled: z.boolean().optional(),
  affiliateCommission: z.coerce.number().min(0).max(100).optional(),
});

interface CreateAgentFormProps {
    form: UseFormReturn<z.infer<typeof formSchema>>;
    step: number;
}

export function CreateAgentForm({ form, step }: CreateAgentFormProps) {
  const { toast } = useToast();
  const affiliateEnabled = form.watch("affiliateEnabled");

  return (
    <>
        {step === 1 && (
            <div className="space-y-8">
                <Card>
                <CardHeader>
                    <CardTitle className="font-headline text-2xl">Agent Identity</CardTitle>
                    <CardDescriptionComponent>Provide the basic information for your agent.</CardDescriptionComponent>
                </CardHeader>
                <CardContent className="space-y-6">
                    <FormField
                    control={form.control}
                    name="agentName"
                    render={({ field }) => (
                        <FormItem>
                        <FormLabel>Agent Name</FormLabel>
                        <FormControl>
                            <Input placeholder="e.g., Data Processor Bot" {...field} />
                        </FormControl>
                        <FormMessage />
                        </FormItem>
                    )}
                    />
                    <FormField
                    control={form.control}
                    name="agentDescription"
                    render={({ field }) => (
                        <FormItem>
                        <FormLabel>Agent Description</FormLabel>
                        <FormControl>
                            <Textarea
                            placeholder="e.g., An AI that receives data from a sheet, processes it with Gemini, and saves it to Drive."
                            {...field}
                            rows={4}
                            />
                        </FormControl>
                         <FormDescription>
                           Describe in detail what your agent does.
                        </FormDescription>
                        <FormMessage />
                        </FormItem>
                    )}
                    />
                     <FormField
                        control={form.control}
                        name="bannerImage"
                        render={({ field }) => (
                            <FormItem>
                                <FormLabel>Agent Banner</FormLabel>
                                <FormDescription>Upload a 600x400px image for your agent's marketplace card.</FormDescription>
                                <FormControl>
                                    <div className="flex flex-col items-center justify-center p-8 border-2 border-dashed rounded-lg text-center cursor-pointer hover:bg-accent/50 transition-colors">
                                        <UploadCloud className="h-12 w-12 text-muted-foreground"/>
                                        <p className="mt-4 text-sm font-semibold text-foreground">Click to upload or drag & drop</p>
                                        <p className="text-xs text-muted-foreground mt-1">PNG, JPG or GIF (600x400px recommended)</p>
                                        <Input type="file" className="sr-only" {...field} />
                                    </div>
                                </FormControl>
                                <FormMessage />
                            </FormItem>
                        )}
                    />
                </CardContent>
                </Card>
            </div>
        )}

        {step === 2 && (
            <div className="space-y-8">
                <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl">Input Channels</CardTitle>
                        <CardDescriptionComponent>Select how your agent will receive information.</CardDescriptionComponent>
                    </CardHeader>
                    <CardContent className="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <FormField control={form.control} name="inputChannels.chatEmbedding" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base">Chat Embedding</FormLabel><FormDescription>Allow the agent to interact on a website.</FormDescription></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="inputChannels.whatsapp" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base">WhatsApp</FormLabel><FormDescription>Connect the agent to a WhatsApp number.</FormDescription></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="inputChannels.websiteUrl" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base">Website URL</FormLabel><FormDescription>Use content from a website as knowledge.</FormDescription></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="inputChannels.googleDrive" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base">Google Drive</FormLabel><FormDescription>Import documents from Google Drive.</FormDescription></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="inputChannels.googleSheets" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base">Google Sheets</FormLabel><FormDescription>Import data from a Google Sheet.</FormDescription></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="inputChannels.json" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base">JSON Data</FormLabel><FormDescription>Provide data via a JSON structure.</FormDescription></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                    </CardContent>
                </Card>
                 <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl">Processing Channels</CardTitle>
                        <CardDescriptionComponent>Select the external tool that will process the information.</CardDescriptionComponent>
                    </CardHeader>
                     <CardContent className="grid grid-cols-1 md:grid-cols-2 gap-6">
                         <FormField control={form.control} name="processingChannels.make" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base flex items-center gap-2"><Share2 className="h-4 w-4"/>Make</FormLabel></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                         <FormField control={form.control} name="processingChannels.zapier" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base flex items-center gap-2"><Share2 className="h-4 w-4"/>Zapier</FormLabel></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                         <FormField control={form.control} name="processingChannels.n8n" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base flex items-center gap-2"><Share2 className="h-4 w-4"/>n8n</FormLabel></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="processingChannels.chatgpt" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base flex items-center gap-2"><Sparkles className="h-4 w-4"/>ChatGPT</FormLabel></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="processingChannels.gemini" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base flex items-center gap-2"><Sparkles className="h-4 w-4"/>Gemini</FormLabel></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="processingChannels.claude" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base flex items-center gap-2"><Sparkles className="h-4 w-4"/>Claude</FormLabel></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="processingChannels.midjourney" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base flex items-center gap-2"><Sparkles className="h-4 w-4"/>Midjourney</FormLabel></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="processingChannels.runway" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base flex items-center gap-2"><Sparkles className="h-4 w-4"/>Runway</FormLabel></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                    </CardContent>
                </Card>
                 <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl">Output Channels</CardTitle>
                        <CardDescriptionComponent>Select how your agent will send information.</CardDescriptionComponent>
                    </CardHeader>
                     <CardContent className="grid grid-cols-1 md:grid-cols-2 gap-6">
                       <FormField control={form.control} name="outputChannels.chatEmbedding" render={({field}) => (
                           <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                               <div className="space-y-0.5"><FormLabel className="text-base">Chat Embedding</FormLabel><FormDescription>Send responses to a website chat.</FormDescription></div>
                               <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                           </FormItem>
                       )}/>
                        <FormField control={form.control} name="outputChannels.whatsapp" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base">WhatsApp</FormLabel><FormDescription>Send messages to a WhatsApp number.</FormDescription></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="outputChannels.websiteUrl" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base">Website URL (Webhook)</FormLabel><FormDescription>Send data to an external URL.</FormDescription></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="outputChannels.googleDrive" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base">Google Drive</FormLabel><FormDescription>Save documents and files to Drive.</FormDescription></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="outputChannels.googleSheets" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base">Google Sheets</FormLabel><FormDescription>Export data to a Google Sheet.</FormDescription></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                        <FormField control={form.control} name="outputChannels.json" render={({field}) => (
                            <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                <div className="space-y-0.5"><FormLabel className="text-base">JSON Data</FormLabel><FormDescription>Return data in a JSON structure.</FormDescription></div>
                                <div className="flex items-center gap-2"><FormControl><Switch checked={field.value} onCheckedChange={field.onChange}/></FormControl><Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button></div>
                            </FormItem>
                        )}/>
                    </CardContent>
                </Card>
            </div>
        )}
        {step === 3 && (
            <div className="space-y-8">
                <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl">Marketplace Page</CardTitle>
                        <CardDescriptionComponent>Configure how your agent will be displayed to customers.</CardDescriptionComponent>
                    </CardHeader>
                    <CardContent className="space-y-6">
                        <FormField
                        control={form.control}
                        name="marketplaceDescription"
                        render={({ field }) => (
                            <FormItem>
                                <FormLabel>Detailed Description</FormLabel>
                                <FormControl>
                                    <div className="w-full rounded-md border border-input bg-background p-3">
                                        {/* This would be a rich text editor */}
                                        <Textarea
                                            placeholder="Describe your agent in detail. Use formatting to highlight features and benefits..."
                                            className="min-h-[200px] border-0 p-0 focus-visible:ring-0 focus-visible:ring-offset-0"
                                            {...field}
                                        />
                                    </div>
                                </FormControl>
                                <FormDescription>
                               This content will be used on your agent's sales page.
                                </FormDescription>
                                <FormMessage />
                            </FormItem>
                        )}
                        />
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                             <FormField
                                control={form.control}
                                name="marketplaceImages"
                                render={({ field }) => (
                                    <FormItem>
                                        <FormLabel>Promotional Images</FormLabel>
                                        <FormControl>
                                            <div className="flex flex-col items-center justify-center p-8 border-2 border-dashed rounded-lg text-center cursor-pointer hover:bg-accent/50 transition-colors">
                                                <ImageIcon className="h-12 w-12 text-muted-foreground"/>
                                                <p className="mt-4 text-sm font-semibold text-foreground">Click to upload images</p>
                                                <p className="text-xs text-muted-foreground mt-1">PNG, JPG, GIF</p>
                                                <Input type="file" multiple className="sr-only" {...field} />
                                            </div>
                                        </FormControl>
                                        <FormMessage />
                                    </FormItem>
                                )}
                            />
                             <FormField
                                control={form.control}
                                name="marketplaceVideos"
                                render={({ field }) => (
                                    <FormItem>
                                        <FormLabel>Demo Video</FormLabel>
                                        <FormControl>
                                            <div className="flex flex-col items-center justify-center p-8 border-2 border-dashed rounded-lg text-center cursor-pointer hover:bg-accent/50 transition-colors">
                                                <Video className="h-12 w-12 text-muted-foreground"/>
                                                <p className="mt-4 text-sm font-semibold text-foreground">Click to upload videos</p>
                                                <p className="text-xs text-muted-foreground mt-1">MP4, WEBM</p>
                                                <Input type="file" multiple className="sr-only" {...field} />
                                            </div>
                                        </FormControl>
                                        <FormMessage />
                                    </FormItem>
                                )}
                            />
                        </div>
                    </CardContent>
                </Card>
                <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl flex items-center gap-2"><UserPlus /> Affiliate Program</CardTitle>
                        <CardDescriptionComponent>
                           Allow others to sell your agent in exchange for a commission.
                        </CardDescriptionComponent>
                    </CardHeader>
                    <CardContent className="space-y-6">
                        <FormField
                            control={form.control}
                            name="affiliateEnabled"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">Enable Affiliation</FormLabel>
                                        <FormDescription>Allow others to promote your agent.</FormDescription>
                                    </div>
                                    <FormControl>
                                        <Switch checked={field.value} onCheckedChange={field.onChange} />
                                    </FormControl>
                                </FormItem>
                            )}
                        />
                        {affiliateEnabled && (
                             <FormField
                                control={form.control}
                                name="affiliateCommission"
                                render={({ field }) => (
                                    <FormItem>
                                    <FormLabel>Affiliate Commission</FormLabel>
                                    <div className="relative">
                                        <FormControl>
                                            <Input type="number" placeholder="20" className="pr-8" {...field} />
                                        </FormControl>
                                        <span className="absolute inset-y-0 right-0 flex items-center pr-3 text-muted-foreground">%</span>
                                    </div>
                                    <FormDescription>The percentage affiliates will earn on each sale.</FormDescription>
                                    <FormMessage />
                                    </FormItem>
                                )}
                            />
                        )}
                    </CardContent>
                </Card>
            </div>
        )}
        {step === 4 && (
             <div className="space-y-8">
                <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl">Pricing</CardTitle>
                        <CardDescriptionComponent>Set a price for your agent.</CardDescriptionComponent>
                    </CardHeader>
                    <CardContent className="space-y-6">
                        <FormField
                        control={form.control}
                        name="pricingModel"
                        render={({ field }) => (
                            <FormItem className="space-y-3">
                            <FormLabel>Pricing Model</FormLabel>
                            <FormControl>
                                <RadioGroup
                                onValueChange={field.onChange}
                                defaultValue={field.value}
                                className="flex flex-col space-y-1"
                                >
                                <FormItem className="flex items-center space-x-3 space-y-0">
                                    <FormControl>
                                    <RadioGroupItem value="lifetime" />
                                    </FormControl>
                                    <FormLabel className="font-normal">
                                    Lifetime Access (one-time payment)
                                    </FormLabel>
                                </FormItem>
                                <FormItem className="flex items-center space-x-3 space-y-0">
                                    <FormControl>
                                    <RadioGroupItem value="subscription" />
                                    </FormControl>
                                    <FormLabel className="font-normal">
                                     Subscription (monthly recurring)
                                    </FormLabel>
                                </FormItem>
                                </RadioGroup>
                            </FormControl>
                            <FormMessage />
                            </FormItem>
                        )}
                        />
                         <FormField
                            control={form.control}
                            name="price"
                            render={({ field }) => (
                                <FormItem>
                                <FormLabel>Price</FormLabel>
                                <div className="relative">
                                    <span className="absolute inset-y-0 left-0 flex items-center pl-3 text-muted-foreground">$</span>
                                    <FormControl>
                                        <Input type="number" placeholder="49.99" className="pl-7" {...field} />
                                    </FormControl>
                                    <span className="absolute inset-y-0 right-0 flex items-center pr-3 text-muted-foreground">
                                        {form.getValues("pricingModel") === "subscription" ? "/ month" : ""}
                                    </span>
                                </div>
                                <FormMessage />
                                </FormItem>
                            )}
                        />
                    </CardContent>
                </Card>
                 <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl flex items-center gap-2"><CreditCard /> Payments</CardTitle>
                        <CardDescriptionComponent>
                            Payments are processed via Stripe.
                        </CardDescriptionComponent>
                    </CardHeader>
                    <CardContent className="space-y-4">
                        <div className="flex items-center justify-between rounded-lg border p-4">
                            <div>
                                <p className="font-semibold">Stripe Connect</p>
                                <p className="text-sm text-muted-foreground">Connected: <span className="text-primary font-medium">acct_...****</span></p>
                            </div>
                            <Button variant="outline">Manage</Button>
                        </div>
                        <div className="flex items-center justify-between rounded-lg border p-4 bg-muted/20">
                           <div className="flex items-center gap-2 text-sm text-muted-foreground">
                             <Percent className="h-4 w-4" />
                             <p>Platform transaction fee</p>
                           </div>
                           <p className="text-sm font-semibold">10%</p>
                        </div>
                    </CardContent>
                </Card>
            </div>
        )}
    </>
  );
}
