"use client";

import { useState } from "react";
import { type UseFormReturn } from "react-hook-form";
import { z } from "zod";
import { generateAgentPersona } from "@/ai/flows/agent-persona-generator";
import { useToast } from "@/hooks/use-toast";

import { Button } from "@/components/ui/button";
import {
  Form,
  FormControl,
  FormDescription,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Sparkles, UploadCloud, Link, FileText, Bot } from "lucide-react";

export const formSchema = z.object({
  agentName: z.string().min(3, "Nome do agente deve ter no mínimo 3 caracteres."),
  agentDescription: z.string().min(10, "Descrição deve ter no mínimo 10 caracteres."),
  persona: z.string().min(10, "Persona deve ter no mínimo 10 caracteres."),
  toneOfVoice: z.string().min(5, "Tom de voz deve ter no mínimo 5 caracteres."),
  knowledgeBaseText: z.string().optional(),
  knowledgeBaseUrl: z.string().url().optional().or(z.literal('')),
  customInstructions: z.string().optional(),
});

interface CreateAgentFormProps {
    form: UseFormReturn<z.infer<typeof formSchema>>;
}

export function CreateAgentForm({ form }: CreateAgentFormProps) {
  const [isGenerating, setIsGenerating] = useState(false);
  const { toast } = useToast();

  const handleGeneratePersona = async () => {
    const agentDescription = form.getValues("agentDescription");
    if (!agentDescription) {
      form.setError("agentDescription", {
        type: "manual",
        message: "Por favor, forneça uma descrição para gerar uma persona.",
      });
      return;
    }

    setIsGenerating(true);
    try {
      const result = await generateAgentPersona({ agentDescription });
      form.setValue("persona", result.persona, { shouldValidate: true });
      form.setValue("toneOfVoice", result.toneOfVoice, { shouldValidate: true });
      toast({
        title: "Persona Gerada!",
        description: "A persona e o tom de voz do seu agente foram criados.",
      });
    } catch (error) {
      console.error("Falha ao gerar persona:", error);
      toast({
        variant: "destructive",
        title: "Falha na Geração",
        description: "Não foi possível gerar uma persona. Tente novamente.",
      });
    } finally {
      setIsGenerating(false);
    }
  };

  function onSubmit(values: z.infer<typeof formSchema>) {
    console.log(values);
    toast({
        title: "Agente Criado!",
        description: "Seu novo agente de IA foi criado e salvo com sucesso.",
    });
  }

  return (
    <Form {...form}>
      <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-8">
        <Card>
          <CardHeader>
            <CardTitle className="font-headline text-2xl">Identidade do Agente</CardTitle>
          </CardHeader>
          <CardContent className="space-y-6">
            <FormField
              control={form.control}
              name="agentName"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>Nome do Agente</FormLabel>
                  <FormControl>
                    <Input placeholder="ex: Mestre de Marketing" {...field} />
                  </FormControl>
                  <FormDescription>
                    Dê ao seu agente um nome único e memorável.
                  </FormDescription>
                  <FormMessage />
                </FormItem>
              )}
            />
             <FormField
              control={form.control}
              name="agentDescription"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>Descrição do Agente</FormLabel>
                  <FormControl>
                    <Textarea
                      placeholder="ex: Uma IA que gera textos criativos para marketing..."
                      {...field}
                    />
                  </FormControl>
                   <FormDescription>
                    Descreva brevemente o que seu agente faz. Isso será usado para gerar a persona.
                  </FormDescription>
                  <FormMessage />
                </FormItem>
              )}
            />
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <div className="flex justify-between items-center">
                <CardTitle className="font-headline text-2xl">Personalidade</CardTitle>
                 <Button type="button" onClick={handleGeneratePersona} disabled={isGenerating}>
                    <Sparkles className="mr-2 h-4 w-4" />
                    {isGenerating ? "Gerando..." : "Gerar com IA"}
                </Button>
            </div>
          </CardHeader>
          <CardContent className="space-y-6">
            <FormField
              control={form.control}
              name="persona"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>Persona</FormLabel>
                  <FormControl>
                    <Textarea
                      placeholder="Descreva a personalidade do seu agente. É formal, espirituoso, empático?"
                      className="min-h-[120px]"
                      {...field}
                    />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="toneOfVoice"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>Tom de Voz</FormLabel>
                  <FormControl>
                    <Textarea
                      placeholder="Defina o estilo de fala do agente. Use frases curtas, emojis, etc."
                      {...field}
                    />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
          </CardContent>
        </Card>
        
        <Card>
          <CardHeader>
            <CardTitle className="font-headline text-2xl">Base de Conhecimento</CardTitle>
             <FormDescription>
              Forneça as informações que seu agente usará para responder às perguntas.
            </FormDescription>
          </CardHeader>
          <CardContent>
             <Tabs defaultValue="text" className="w-full">
              <TabsList className="grid w-full grid-cols-3">
                <TabsTrigger value="text"><FileText className="mr-2 h-4 w-4"/>Texto</TabsTrigger>
                <TabsTrigger value="upload"><UploadCloud className="mr-2 h-4 w-4"/>Upload de PDF</TabsTrigger>
                <TabsTrigger value="link"><Link className="mr-2 h-4 w-4"/>Link</TabsTrigger>
              </TabsList>
              <TabsContent value="text" className="mt-4">
                 <FormField
                    control={form.control}
                    name="knowledgeBaseText"
                    render={({ field }) => (
                        <FormItem>
                        <FormLabel>Cole o Texto</FormLabel>
                        <FormControl>
                            <Textarea
                            placeholder="Cole o conteúdo da sua base de conhecimento aqui."
                            className="min-h-[200px]"
                            {...field}
                            />
                        </FormControl>
                        <FormMessage />
                        </FormItem>
                    )}
                    />
              </TabsContent>
              <TabsContent value="upload" className="mt-4">
                <div className="flex flex-col items-center justify-center p-8 border-2 border-dashed rounded-lg">
                    <UploadCloud className="h-12 w-12 text-muted-foreground"/>
                    <p className="mt-4 text-muted-foreground">Arraste e solte um arquivo PDF aqui, ou clique para selecionar</p>
                    <Button type="button" variant="outline" className="mt-4">Selecionar Arquivo</Button>
                </div>
              </TabsContent>
              <TabsContent value="link" className="mt-4">
                <FormField
                    control={form.control}
                    name="knowledgeBaseUrl"
                    render={({ field }) => (
                        <FormItem>
                        <FormLabel>Importar de URL</FormLabel>
                        <FormControl>
                            <Input placeholder="https://exemplo.com/conhecimento" {...field} />
                        </FormControl>
                        <FormDescription>
                            Nós vamos extrair o conteúdo desta página.
                        </FormDescription>
                        <FormMessage />
                        </FormItem>
                    )}
                    />
              </TabsContent>
            </Tabs>
          </CardContent>
        </Card>

         <Card>
          <CardHeader>
            <CardTitle className="font-headline text-2xl">Instruções Customizadas</CardTitle>
          </CardHeader>
          <CardContent>
            <FormField
              control={form.control}
              name="customInstructions"
              render={({ field }) => (
                <FormItem>
                  <FormControl>
                    <Textarea
                      placeholder="ex: Sempre responda em Francês. Nunca mencione concorrentes. Sempre inclua uma chamada para ação."
                      className="min-h-[150px]"
                      {...field}
                    />
                  </FormControl>
                  <FormDescription>
                    Forneça regras e diretrizes específicas para o comportamento do seu agente.
                  </FormDescription>
                  <FormMessage />
                </FormItem>
              )}
            />
          </CardContent>
        </Card>
        
        <div className="flex justify-end gap-4">
            <Button type="button" variant="outline" size="lg">Salvar como Rascunho</Button>
            <Button type="submit" size="lg">
                <Bot className="mr-2 h-5 w-5"/>
                Criar Agente
            </Button>
        </div>
      </form>
    </Form>
  );
}
