"use client";

import { useState, useEffect, useRef } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { Send, Bot, User } from "lucide-react";
import { Avatar, AvatarFallback } from "@/components/ui/avatar";
import { cn } from '@/lib/utils';

interface AgentTestChatProps {
    agentName: string;
    persona: string;
}

interface Message {
    sender: 'user' | 'agent';
    text: string;
}

export function AgentTestChat({ agentName, persona }: AgentTestChatProps) {
    const [messages, setMessages] = useState<Message[]>([]);
    const [input, setInput] = useState('');
    const messagesEndRef = useRef<HTMLDivElement>(null);

    const agentDisplayName = agentName || "Agente";

    useEffect(() => {
        setMessages([
            { sender: 'agent', text: `Olá! Eu sou ${agentDisplayName}. Como posso te ajudar hoje?` }
        ]);
    }, [agentDisplayName]);

    useEffect(() => {
        messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' });
    }, [messages]);


    const handleSend = () => {
        if (input.trim() === '') return;

        const userMessage: Message = { sender: 'user', text: input };
        
        const agentResponseText = persona 
            ? `(Assumindo a persona: "${persona.substring(0, 50)}...")\n\nObrigado por sua mensagem: "${input}"` 
            : `Obrigado por sua mensagem: "${input}"`;

        const agentMessage: Message = { sender: 'agent', text: agentResponseText };

        setMessages(prev => [...prev, userMessage, agentMessage]);
        setInput('');
    };

    return (
        <Card className="flex flex-col h-full max-h-[1620px]">
            <CardHeader>
                <CardTitle className="font-headline text-2xl flex items-center gap-2">
                    <Bot />
                    Chat de Teste do Agente
                </CardTitle>
            </CardHeader>
            <CardContent className="flex flex-col flex-grow">
                <div className="flex-grow space-y-4 overflow-y-auto p-4 border rounded-lg bg-muted/30 min-h-[400px]">
                    {messages.map((msg, index) => (
                        <div key={index} className={cn("flex items-end gap-2", msg.sender === 'user' ? "justify-end" : "justify-start")}>
                             {msg.sender === 'agent' && (
                                <Avatar className="h-8 w-8 bg-primary text-primary-foreground">
                                    <AvatarFallback>{agentDisplayName.charAt(0)}</AvatarFallback>
                                </Avatar>
                            )}
                            <div
                                className={cn(
                                    "rounded-lg px-4 py-2 max-w-[80%] whitespace-pre-wrap",
                                    msg.sender === 'user' 
                                        ? "bg-primary text-primary-foreground" 
                                        : "bg-background"
                                )}
                            >
                                {msg.text}
                            </div>
                             {msg.sender === 'user' && (
                                <Avatar className="h-8 w-8 bg-muted text-muted-foreground">
                                    <AvatarFallback><User size={16} /></AvatarFallback>
                                </Avatar>
                            )}
                        </div>
                    ))}
                    <div ref={messagesEndRef} />
                </div>
                <div className="mt-4 flex gap-2">
                    <Input
                        placeholder="Digite sua mensagem..."
                        value={input}
                        onChange={(e) => setInput(e.target.value)}
                        onKeyDown={(e) => e.key === 'Enter' && handleSend()}
                    />
                    <Button onClick={handleSend}><Send size={16} /></Button>
                </div>
            </CardContent>
        </Card>
    );
}
