"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.RuntimeManager = exports.GENKIT_REFLECTION_API_SPEC_VERSION = void 0;
const axios_1 = __importDefault(require("axios"));
const chokidar_1 = __importDefault(require("chokidar"));
const events_1 = __importDefault(require("events"));
const promises_1 = __importDefault(require("fs/promises"));
const path_1 = __importDefault(require("path"));
const action_1 = require("../types/action");
const apis = __importStar(require("../types/apis"));
const logger_1 = require("../utils/logger");
const utils_1 = require("../utils/utils");
const types_1 = require("./types");
const STREAM_DELIMITER = '\n';
const HEALTH_CHECK_INTERVAL = 5000;
exports.GENKIT_REFLECTION_API_SPEC_VERSION = 1;
class RuntimeManager {
    telemetryServerUrl;
    manageHealth;
    filenameToRuntimeMap = {};
    filenameToDevUiMap = {};
    idToFileMap = {};
    eventEmitter = new events_1.default();
    constructor(telemetryServerUrl, manageHealth = true) {
        this.telemetryServerUrl = telemetryServerUrl;
        this.manageHealth = manageHealth;
    }
    static async create(options) {
        const manager = new RuntimeManager(options.telemetryServerUrl, options.manageHealth ?? true);
        await manager.setupRuntimesWatcher();
        await manager.setupDevUiWatcher();
        if (manager.manageHealth) {
            setInterval(async () => await manager.performHealthChecks(), HEALTH_CHECK_INTERVAL);
        }
        return manager;
    }
    listRuntimes() {
        return Object.fromEntries(Object.values(this.filenameToRuntimeMap).map((runtime) => [
            runtime.id,
            runtime,
        ]));
    }
    getRuntimeById(id) {
        const fileName = this.idToFileMap[id];
        return fileName ? this.filenameToRuntimeMap[fileName] : undefined;
    }
    getMostRecentRuntime() {
        const runtimes = Object.values(this.filenameToRuntimeMap);
        return runtimes.length === 0
            ? undefined
            : runtimes.reduce((a, b) => new Date(a.timestamp) > new Date(b.timestamp) ? a : b);
    }
    getMostRecentDevUI() {
        const toolsInfo = Object.values(this.filenameToDevUiMap);
        return toolsInfo.length === 0
            ? undefined
            : toolsInfo.reduce((a, b) => new Date(a.timestamp) > new Date(b.timestamp) ? a : b);
    }
    onRuntimeEvent(listener) {
        Object.values(types_1.RuntimeEvent).forEach((event) => this.eventEmitter.on(event, (rt) => listener(event, rt)));
    }
    async listActions() {
        const runtime = this.getMostRecentRuntime();
        if (!runtime) {
            throw new Error('No runtimes found');
        }
        const response = await axios_1.default
            .get(`${runtime.reflectionServerUrl}/api/actions`)
            .catch((err) => this.httpErrorHandler(err, 'Error listing actions.'));
        return response.data;
    }
    async runAction(input, streamingCallback) {
        const runtime = this.getMostRecentRuntime();
        if (!runtime) {
            throw new Error('No runtimes found');
        }
        if (streamingCallback) {
            const response = await axios_1.default
                .post(`${runtime.reflectionServerUrl}/api/runAction?stream=true`, input, {
                headers: {
                    'Content-Type': 'application/json',
                },
                responseType: 'stream',
            })
                .catch(this.httpErrorHandler);
            let genkitVersion;
            if (response.headers['x-genkit-version']) {
                genkitVersion = response.headers['x-genkit-version'];
            }
            const stream = response.data;
            let buffer = '';
            stream.on('data', (data) => {
                buffer += data;
                while (buffer.includes(STREAM_DELIMITER)) {
                    try {
                        streamingCallback(JSON.parse(buffer.substring(0, buffer.indexOf(STREAM_DELIMITER))));
                        buffer = buffer.substring(buffer.indexOf(STREAM_DELIMITER) + STREAM_DELIMITER.length);
                    }
                    catch (err) {
                        logger_1.logger.error(`Bad stream: ${err}`);
                        break;
                    }
                }
            });
            let resolver;
            let rejecter;
            const promise = new Promise((resolve, reject) => {
                resolver = resolve;
                rejecter = reject;
            });
            stream.on('end', () => {
                const parsedBuffer = JSON.parse(buffer);
                if (parsedBuffer.error) {
                    const err = new types_1.GenkitToolsError(`Error running action key='${input.key}'.`);
                    err.data = {
                        ...parsedBuffer.error,
                        stack: (parsedBuffer.error?.details).stack,
                        data: {
                            genkitErrorMessage: parsedBuffer.error?.message,
                            genkitErrorDetails: parsedBuffer.error?.details,
                        },
                    };
                    rejecter(err);
                    return;
                }
                const actionResponse = action_1.RunActionResponseSchema.parse(parsedBuffer);
                if (genkitVersion) {
                    actionResponse.genkitVersion = genkitVersion;
                }
                resolver(actionResponse);
            });
            stream.on('error', (err) => {
                rejecter(err);
            });
            return promise;
        }
        else {
            const response = await axios_1.default
                .post(`${runtime.reflectionServerUrl}/api/runAction`, input, {
                headers: {
                    'Content-Type': 'application/json',
                },
            })
                .catch((err) => this.httpErrorHandler(err, `Error running action key='${input.key}'.`));
            const resp = action_1.RunActionResponseSchema.parse(response.data);
            if (response.headers['x-genkit-version']) {
                resp.genkitVersion = response.headers['x-genkit-version'];
            }
            return resp;
        }
    }
    async listTraces(input) {
        const { limit, continuationToken, filter } = input;
        let query = '';
        if (limit) {
            query += `limit=${limit}`;
        }
        if (continuationToken) {
            if (query !== '') {
                query += '&';
            }
            query += `continuationToken=${continuationToken}`;
        }
        if (filter) {
            if (query !== '') {
                query += '&';
            }
            query += `filter=${encodeURI(JSON.stringify(filter))}`;
        }
        const response = await axios_1.default
            .get(`${this.telemetryServerUrl}/api/traces?${query}`)
            .catch((err) => this.httpErrorHandler(err, `Error listing traces for query='${query}'.`));
        return apis.ListTracesResponseSchema.parse(response.data);
    }
    async getTrace(input) {
        const { traceId } = input;
        const response = await axios_1.default
            .get(`${this.telemetryServerUrl}/api/traces/${traceId}`)
            .catch((err) => this.httpErrorHandler(err, `Error getting trace for traceId='${traceId}'`));
        return response.data;
    }
    async notifyRuntime(runtime) {
        try {
            await axios_1.default.post(`${runtime.reflectionServerUrl}/api/notify`, {
                telemetryServerUrl: this.telemetryServerUrl,
                reflectionApiSpecVersion: exports.GENKIT_REFLECTION_API_SPEC_VERSION,
            });
        }
        catch (error) {
            logger_1.logger.error(`Failed to notify runtime ${runtime.id}: ${error}`);
        }
    }
    async setupRuntimesWatcher() {
        try {
            const runtimesDir = await (0, utils_1.findRuntimesDir)();
            await promises_1.default.mkdir(runtimesDir, { recursive: true });
            const watcher = chokidar_1.default.watch(runtimesDir, {
                persistent: true,
                ignoreInitial: false,
            });
            watcher.on('add', (filePath) => this.handleNewRuntime(filePath));
            if (this.manageHealth) {
                watcher.on('unlink', (filePath) => this.handleRemovedRuntime(filePath));
            }
            for (const runtime of await promises_1.default.readdir(runtimesDir)) {
                await this.handleNewRuntime(path_1.default.resolve(runtimesDir, runtime));
            }
        }
        catch (error) {
            logger_1.logger.error('Failed to set up runtimes watcher:', error);
        }
    }
    async setupDevUiWatcher() {
        try {
            const serversDir = await (0, utils_1.findServersDir)();
            await promises_1.default.mkdir(serversDir, { recursive: true });
            const watcher = chokidar_1.default.watch(serversDir, {
                persistent: true,
                ignoreInitial: false,
            });
            watcher.on('add', (filePath) => this.handleNewDevUi(filePath));
            if (this.manageHealth) {
                watcher.on('unlink', (filePath) => this.handleRemovedDevUi(filePath));
            }
            for (const toolsInfo of await promises_1.default.readdir(serversDir)) {
                await this.handleNewDevUi(path_1.default.resolve(serversDir, toolsInfo));
            }
        }
        catch (error) {
            logger_1.logger.error('Failed to set up tools server watcher:', error);
        }
    }
    async handleNewDevUi(filePath) {
        try {
            const { content, toolsInfo } = await (0, utils_1.retriable)(async () => {
                const content = await promises_1.default.readFile(filePath, 'utf-8');
                const toolsInfo = JSON.parse(content);
                return { content, toolsInfo };
            }, { maxRetries: 10, delayMs: 500 });
            if ((0, utils_1.isValidDevToolsInfo)(toolsInfo)) {
                const fileName = path_1.default.basename(filePath);
                if (await (0, utils_1.checkServerHealth)(toolsInfo.url)) {
                    this.filenameToDevUiMap[fileName] = toolsInfo;
                }
                else {
                    logger_1.logger.debug('Found an unhealthy tools config file', fileName);
                    await (0, utils_1.removeToolsInfoFile)(fileName);
                }
            }
            else {
                logger_1.logger.error(`Unexpected file in the servers directory: ${content}`);
            }
        }
        catch (error) {
            logger_1.logger.info('Error reading tools config', error);
            return undefined;
        }
    }
    handleRemovedDevUi(filePath) {
        const fileName = path_1.default.basename(filePath);
        if (fileName in this.filenameToDevUiMap) {
            const toolsInfo = this.filenameToDevUiMap[fileName];
            delete this.filenameToDevUiMap[fileName];
            logger_1.logger.debug(`Removed Dev UI with url ${toolsInfo.url}.`);
        }
    }
    async handleNewRuntime(filePath) {
        try {
            const { content, runtimeInfo } = await (0, utils_1.retriable)(async () => {
                const content = await promises_1.default.readFile(filePath, 'utf-8');
                const runtimeInfo = JSON.parse(content);
                runtimeInfo.projectName = (0, utils_1.projectNameFromGenkitFilePath)(filePath);
                return { content, runtimeInfo };
            }, { maxRetries: 10, delayMs: 500 });
            if (isValidRuntimeInfo(runtimeInfo)) {
                const fileName = path_1.default.basename(filePath);
                if (await (0, utils_1.checkServerHealth)(runtimeInfo.reflectionServerUrl)) {
                    if (runtimeInfo.reflectionApiSpecVersion !=
                        exports.GENKIT_REFLECTION_API_SPEC_VERSION) {
                        if (!runtimeInfo.reflectionApiSpecVersion ||
                            runtimeInfo.reflectionApiSpecVersion <
                                exports.GENKIT_REFLECTION_API_SPEC_VERSION) {
                            logger_1.logger.warn('Genkit CLI is newer than runtime library. Some feature may not be supported. ' +
                                'Consider upgrading your runtime library version (debug info: expected ' +
                                `${exports.GENKIT_REFLECTION_API_SPEC_VERSION}, got ${runtimeInfo.reflectionApiSpecVersion}).`);
                        }
                        else {
                            logger_1.logger.error('Genkit CLI version is outdated. Please update `genkit-cli` to the latest version.');
                            process.exit(1);
                        }
                    }
                    this.filenameToRuntimeMap[fileName] = runtimeInfo;
                    this.idToFileMap[runtimeInfo.id] = fileName;
                    this.eventEmitter.emit(types_1.RuntimeEvent.ADD, runtimeInfo);
                    await this.notifyRuntime(runtimeInfo);
                    logger_1.logger.debug(`Added runtime with ID ${runtimeInfo.id} at URL: ${runtimeInfo.reflectionServerUrl}`);
                }
                else {
                    await this.removeRuntime(fileName);
                }
            }
            else {
                logger_1.logger.error(`Unexpected file in the runtimes directory: ${content}`);
            }
        }
        catch (error) {
            logger_1.logger.error(`Error processing file ${filePath}:`, error);
        }
    }
    handleRemovedRuntime(filePath) {
        const fileName = path_1.default.basename(filePath);
        if (fileName in this.filenameToRuntimeMap) {
            const runtime = this.filenameToRuntimeMap[fileName];
            delete this.filenameToRuntimeMap[fileName];
            delete this.idToFileMap[runtime.id];
            this.eventEmitter.emit(types_1.RuntimeEvent.REMOVE, runtime);
            logger_1.logger.debug(`Removed runtime with id ${runtime.id}.`);
        }
    }
    httpErrorHandler(error, message) {
        const newError = new types_1.GenkitToolsError(message || 'Internal Error');
        if (error.response) {
            if ((error.response?.data).message) {
                newError.message = (error.response?.data).message;
            }
            newError.data = error.response.data;
            throw newError;
        }
        throw new types_1.GenkitToolsError(message || 'Internal Error', {
            cause: error.cause,
        });
    }
    async performHealthChecks() {
        const healthCheckPromises = Object.entries(this.filenameToRuntimeMap).map(async ([fileName, runtime]) => {
            if (!(await (0, utils_1.checkServerHealth)(runtime.reflectionServerUrl))) {
                await this.removeRuntime(fileName);
            }
        });
        return Promise.all(healthCheckPromises);
    }
    async removeRuntime(fileName) {
        const runtime = this.filenameToRuntimeMap[fileName];
        if (runtime) {
            try {
                const runtimesDir = await (0, utils_1.findRuntimesDir)();
                const runtimeFilePath = path_1.default.join(runtimesDir, fileName);
                await promises_1.default.unlink(runtimeFilePath);
            }
            catch (error) {
                logger_1.logger.debug(`Failed to delete runtime file: ${error}`);
            }
            logger_1.logger.debug(`Removed unhealthy runtime with ID ${runtime.id} from manager.`);
        }
    }
}
exports.RuntimeManager = RuntimeManager;
function isValidRuntimeInfo(data) {
    let timestamp = '';
    if (typeof data.timestamp === 'string') {
        timestamp = data.timestamp.replaceAll('_', ':');
    }
    return (typeof data === 'object' &&
        typeof data.id === 'string' &&
        typeof data.pid === 'number' &&
        typeof data.reflectionServerUrl === 'string' &&
        typeof data.timestamp === 'string' &&
        !isNaN(Date.parse(timestamp)));
}
//# sourceMappingURL=manager.js.map